<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class Setting extends Model
{
    protected $fillable = [
        'key',
        'value',
        'type',
        'group',
        'label',
        'description',
        'sort_order'
    ];

    protected $casts = [
        'sort_order' => 'integer',
    ];

    /**
     * Ayar değerini al
     */
    public static function get($key, $default = null)
    {
        return Cache::remember("setting_{$key}", 3600, function () use ($key, $default) {
            $setting = self::where('key', $key)->first();
            return $setting ? $setting->value : $default;
        });
    }

    /**
     * Ayar değerini kaydet
     */
    public static function set($key, $value)
    {
        $setting = self::updateOrCreate(
            ['key' => $key],
            ['value' => $value]
        );

        // Cache'i temizle
        Cache::forget("setting_{$key}");

        return $setting;
    }

    /**
     * Grup bazında ayarları al
     */
    public static function getByGroup($group)
    {
        return Cache::remember("settings_group_{$group}", 3600, function () use ($group) {
            return self::where('group', $group)
                      ->orderBy('sort_order')
                      ->orderBy('label')
                      ->get()
                      ->pluck('value', 'key')
                      ->toArray();
        });
    }

    /**
     * Tüm ayarları al
     */
    public static function getAll()
    {
        return Cache::remember('all_settings', 3600, function () {
            return self::all()->pluck('value', 'key')->toArray();
        });
    }

    /**
     * Cache'i temizle
     */
    public static function clearCache()
    {
        Cache::flush();
    }

    /**
     * Scope: Belirli grup
     */
    public function scopeByGroup($query, $group)
    {
        return $query->where('group', $group);
    }

    /**
     * Scope: Sıralı
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order')->orderBy('label');
    }

    /**
     * Boolean değer kontrolü
     */
    public function getBooleanValueAttribute()
    {
        if ($this->type === 'boolean') {
            return in_array(strtolower($this->value), ['1', 'true', 'on', 'yes']);
        }
        return $this->value;
    }
}
