<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Crypt;

class PaymentSetting extends Model
{
    protected $fillable = [
        'key',
        'value',
        'type',
        'group',
        'label',
        'description',
        'is_encrypted'
    ];

    protected $casts = [
        'is_encrypted' => 'boolean',
    ];

    /**
     * Ayar değerini al (şifrelenmiş ise çöz)
     */
    public function getValueAttribute($value)
    {
        if ($this->is_encrypted && $value) {
            try {
                return Crypt::decryptString($value);
            } catch (\Exception $e) {
                return $value;
            }
        }

        return $value;
    }

    /**
     * Ayar değerini kaydet (şifrelenmiş ise şifrele)
     */
    public function setValueAttribute($value)
    {
        if ($this->is_encrypted && $value) {
            $this->attributes['value'] = Crypt::encryptString($value);
        } else {
            $this->attributes['value'] = $value;
        }
    }

    /**
     * Ayar değerini al (static method)
     */
    public static function get($key, $default = null)
    {
        $setting = self::where('key', $key)->first();

        if (!$setting) {
            return $default;
        }

        // Boolean değerler için
        if ($setting->type === 'boolean') {
            return (bool) $setting->value;
        }

        // Number değerler için
        if ($setting->type === 'number') {
            return (float) $setting->value;
        }

        return $setting->value;
    }

    /**
     * Ayar değerini kaydet (static method)
     */
    public static function set($key, $value)
    {
        $setting = self::where('key', $key)->first();

        if ($setting) {
            $setting->value = $value;
            $setting->save();
        } else {
            self::create([
                'key' => $key,
                'value' => $value,
                'type' => 'text',
                'group' => 'general',
                'label' => $key,
            ]);
        }

        return true;
    }

    /**
     * Grup bazında ayarları al
     */
    public static function getByGroup($group)
    {
        return self::where('group', $group)->get()->keyBy('key');
    }

    /**
     * PayTR ayarlarını al
     */
    public static function getPayTRSettings()
    {
        return [
            'enabled' => self::get('paytr_enabled', false),
            'test_mode' => self::get('paytr_test_mode', true),
            'merchant_id' => self::get('paytr_merchant_id', ''),
            'merchant_key' => self::get('paytr_merchant_key', ''),
            'merchant_salt' => self::get('paytr_merchant_salt', ''),
        ];
    }
}
