<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Order;
use App\Services\PayTRService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PaymentController extends Controller
{
    private $paytrService;

    public function __construct(PayTRService $paytrService)
    {
        $this->paytrService = $paytrService;
    }

    /**
     * Kredi kartı ödeme sayfası
     */
    public function creditCard(Request $request)
    {
        $orderId = $request->get('order_id');

        if (!$orderId) {
            return redirect()->route('cart.index')->with('error', 'Geçersiz sipariş!');
        }

        $order = Order::where('id', $orderId)
                     ->where('user_id', Auth::id())
                     ->with('items.product')
                     ->firstOrFail();

        // Sipariş zaten ödenmiş mi kontrol et
        if ($order->payment_status === 'paid') {
            return redirect()->route('checkout.success', $order->order_number)
                           ->with('info', 'Bu sipariş zaten ödenmiş!');
        }

        // PayTR aktif mi kontrol et
        if (!$this->paytrService->isEnabled()) {
            return redirect()->route('checkout.index')
                           ->with('error', 'Kredi kartı ödemesi şu anda kullanılamıyor!');
        }

        try {
            // PayTR token oluştur
            $paymentData = $this->paytrService->createPaymentToken($order, [
                'name' => Auth::user()->name,
                'email' => Auth::user()->email,
                'phone' => $order->delivery_phone,
                'address' => $order->delivery_address
            ]);

            // Ödeme deneme zamanını kaydet
            $order->update(['payment_attempted_at' => now()]);

            return view('payment.credit-card', compact('order', 'paymentData'));

        } catch (\Exception $e) {
            Log::error('PayTR Token Error: ' . $e->getMessage());
            return redirect()->route('checkout.index')
                           ->with('error', 'Ödeme işlemi başlatılamadı: ' . $e->getMessage());
        }
    }

    /**
     * PayTR başarılı ödeme callback
     */
    public function success(Request $request)
    {
        return view('payment.success');
    }

    /**
     * PayTR başarısız ödeme callback
     */
    public function fail(Request $request)
    {
        return view('payment.fail');
    }

    /**
     * PayTR IPN (Instant Payment Notification) callback
     */
    public function callback(Request $request)
    {
        try {
            // PayTR'den gelen verileri doğrula
            $result = $this->paytrService->verifyCallback($request->all());

            if (!$result['success']) {
                Log::error('PayTR Callback Verification Failed', $request->all());
                return response('FAIL', 200);
            }

            // Siparişi bul
            $order = Order::where('order_number', $result['order_number'])->first();

            if (!$order) {
                Log::error('PayTR Callback: Order not found', ['order_number' => $result['order_number']]);
                return response('FAIL', 200);
            }

            DB::beginTransaction();

            try {
                // Ödeme durumuna göre sipariş güncelle
                if ($result['status'] === 'success') {
                    $order->update([
                        'payment_status' => 'paid',
                        'paid_at' => now(),
                        'status' => 'confirmed',
                        'payment_response' => $request->all()
                    ]);

                    Log::info('PayTR Payment Success', [
                        'order_number' => $order->order_number,
                        'amount' => $result['total_amount']
                    ]);
                } else {
                    $order->update([
                        'payment_status' => 'failed',
                        'payment_response' => $request->all()
                    ]);

                    // Stokları geri ekle
                    foreach ($order->items as $item) {
                        $item->product->increment('stock_quantity', $item->quantity);
                    }

                    Log::info('PayTR Payment Failed', [
                        'order_number' => $order->order_number,
                        'reason' => $request->get('failed_reason_msg', 'Bilinmeyen hata')
                    ]);
                }

                DB::commit();
                return response('OK', 200);

            } catch (\Exception $e) {
                DB::rollback();
                Log::error('PayTR Callback Database Error: ' . $e->getMessage());
                return response('FAIL', 200);
            }

        } catch (\Exception $e) {
            Log::error('PayTR Callback Error: ' . $e->getMessage());
            return response('FAIL', 200);
        }
    }

    /**
     * PayTR durumu kontrol et (AJAX)
     */
    public function status(Request $request)
    {
        if (!Auth::check()) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        $orderId = $request->get('order_id');
        $order = Order::where('id', $orderId)
                     ->where('user_id', Auth::id())
                     ->first();

        if (!$order) {
            return response()->json(['error' => 'Order not found'], 404);
        }

        return response()->json([
            'payment_status' => $order->payment_status,
            'status' => $order->status,
            'paid_at' => $order->paid_at?->format('d.m.Y H:i'),
            'order_number' => $order->order_number
        ]);
    }
}
