<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Category;
use App\Models\Product;
use App\Models\Slider;
use App\Models\Story;

class HomeController extends Controller
{
    /**
     * Ana sayfa görünümünü döndürür
     */
    public function index()
    {
        // Slider'ları çek
        $sliders = Slider::active()
            ->ordered()
            ->take(5)
            ->get();

        // Story'leri çek
        $stories = Story::active()
            ->ordered()
            ->take(10)
            ->get();

        // Ana kategorileri çek
        $categories = Category::active()
            ->main()
            ->ordered()
            ->take(6)
            ->get();

        // Öne çıkan ürünleri çek
        $featuredProducts = Product::active()
            ->featured()
            ->inStock()
            ->with('category')
            ->take(8)
            ->get();

        // Yeni ürünleri çek
        $newProducts = Product::active()
            ->inStock()
            ->with('category')
            ->latest()
            ->take(8)
            ->get();

        // İndirimli ürünleri çek
        $saleProducts = Product::active()
            ->onSale()
            ->inStock()
            ->with('category')
            ->take(8)
            ->get();

        return view('home', compact('sliders', 'stories', 'categories', 'featuredProducts', 'newProducts', 'saleProducts'));
    }

    /**
     * Hakkımızda sayfası
     */
    public function about()
    {
        // Hakkımızda sayfası aktif mi kontrol et
        if (!\App\Models\Setting::get('about_page_enabled', true)) {
            abort(404, 'Sayfa bulunamadı.');
        }

        return view('about');
    }

    /**
     * Story detay sayfası
     */
    public function showStory(Story $story)
    {
        // Story aktif mi kontrol et
        if (!$story->is_active || ($story->expires_at && $story->expires_at < now())) {
            abort(404, 'Story bulunamadı.');
        }

        // Diğer aktif story'leri al (navigasyon için)
        $allStories = Story::active()->ordered()->get();
        $currentIndex = $allStories->search(function($item) use ($story) {
            return $item->id === $story->id;
        });

        $previousStory = $currentIndex > 0 ? $allStories[$currentIndex - 1] : null;
        $nextStory = $currentIndex < $allStories->count() - 1 ? $allStories[$currentIndex + 1] : null;

        return view('story.show', compact('story', 'allStories', 'previousStory', 'nextStory', 'currentIndex'));
    }
}
