<?php

namespace App\Services;

use App\Models\PaymentSetting;
use App\Models\Order;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class PayTRService
{
    private $merchantId;
    private $merchantKey;
    private $merchantSalt;
    private $testMode;
    private $enabled;

    public function __construct()
    {
        $settings = PaymentSetting::getPayTRSettings();
        
        $this->enabled = $settings['enabled'];
        $this->testMode = $settings['test_mode'];
        $this->merchantId = $settings['merchant_id'];
        $this->merchantKey = $settings['merchant_key'];
        $this->merchantSalt = $settings['merchant_salt'];
    }

    /**
     * PayTR aktif mi kontrol et
     */
    public function isEnabled(): bool
    {
        return $this->enabled && !empty($this->merchantId) && !empty($this->merchantKey) && !empty($this->merchantSalt);
    }

    /**
     * Test modu aktif mi
     */
    public function isTestMode(): bool
    {
        return $this->testMode;
    }

    /**
     * Ödeme formu için token oluştur
     */
    public function createPaymentToken(Order $order, array $userInfo = []): array
    {
        if (!$this->isEnabled()) {
            throw new \Exception('PayTR ödeme sistemi aktif değil!');
        }

        // Sipariş öğelerini hazırla
        $basket = [];
        $basketStr = '';
        
        foreach ($order->items as $item) {
            $basket[] = [
                $item->product_name,
                number_format($item->unit_price, 2, '.', ''),
                $item->quantity
            ];
            $basketStr .= [$item->product_name, number_format($item->unit_price, 2, '.', ''), $item->quantity];
        }

        // Kullanıcı bilgileri
        $userBasket = base64_encode(json_encode($basket));
        $userIp = request()->ip();
        $emailAddress = $userInfo['email'] ?? $order->user->email ?? 'test@test.com';
        
        // PayTR parametreleri
        $merchantOid = $order->order_number . '_' . time();
        $paymentAmount = number_format($order->total_amount * 100, 0, '', ''); // Kuruş cinsinden
        $currency = 'TL';
        $testMode = $this->testMode ? '1' : '0';
        
        // Callback URL'leri
        $okUrl = route('payment.success');
        $failUrl = route('payment.fail');
        
        // Hash oluştur
        $hashStr = $this->merchantId . $userIp . $merchantOid . $emailAddress . $paymentAmount . $userBasket . $testMode . $this->merchantSalt;
        $token = base64_encode(hash_hmac('sha256', $hashStr, $this->merchantKey, true));

        // PayTR'ye gönderilecek veriler
        $postData = [
            'merchant_id' => $this->merchantId,
            'user_ip' => $userIp,
            'merchant_oid' => $merchantOid,
            'email' => $emailAddress,
            'payment_amount' => $paymentAmount,
            'paytr_token' => $token,
            'user_basket' => $userBasket,
            'debug_on' => $testMode,
            'no_installment' => '0',
            'max_installment' => '0',
            'user_name' => $userInfo['name'] ?? $order->delivery_name,
            'user_address' => $userInfo['address'] ?? $order->delivery_address,
            'user_phone' => $userInfo['phone'] ?? $order->delivery_phone,
            'merchant_ok_url' => $okUrl,
            'merchant_fail_url' => $failUrl,
            'timeout_limit' => '30',
            'currency' => $currency,
            'test_mode' => $testMode,
        ];

        // PayTR API'ye istek gönder
        $result = $this->sendRequest('https://www.paytr.com/odeme/api/get-token', $postData);

        if ($result['status'] === 'success') {
            // Siparişe PayTR token'ını kaydet
            $order->update([
                'payment_token' => $result['token'],
                'paytr_merchant_oid' => $merchantOid
            ]);

            return [
                'success' => true,
                'token' => $result['token'],
                'merchant_oid' => $merchantOid,
                'iframe_url' => 'https://www.paytr.com/odeme/guvenli/' . $result['token']
            ];
        } else {
            Log::error('PayTR Token Error: ' . $result['reason']);
            throw new \Exception('Ödeme token\'ı oluşturulamadı: ' . $result['reason']);
        }
    }

    /**
     * PayTR callback'ini doğrula
     */
    public function verifyCallback(array $postData): array
    {
        $merchantOid = $postData['merchant_oid'] ?? '';
        $status = $postData['status'] ?? '';
        $totalAmount = $postData['total_amount'] ?? '';
        $hash = $postData['hash'] ?? '';

        // Hash doğrulama
        $hashStr = $merchantOid . $this->merchantSalt . $status . $totalAmount;
        $calculatedHash = base64_encode(hash_hmac('sha256', $hashStr, $this->merchantKey, true));

        if ($hash !== $calculatedHash) {
            Log::error('PayTR Callback Hash Mismatch', [
                'received_hash' => $hash,
                'calculated_hash' => $calculatedHash,
                'merchant_oid' => $merchantOid
            ]);
            
            return [
                'success' => false,
                'error' => 'Hash doğrulaması başarısız'
            ];
        }

        // Sipariş numarasını çıkar
        $orderNumber = explode('_', $merchantOid)[0];
        
        return [
            'success' => true,
            'order_number' => $orderNumber,
            'status' => $status,
            'total_amount' => $totalAmount / 100, // TL cinsine çevir
            'merchant_oid' => $merchantOid
        ];
    }

    /**
     * HTTP isteği gönder
     */
    private function sendRequest(string $url, array $data): array
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
        curl_setopt($ch, CURLOPT_FRESH_CONNECT, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 20);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

        $result = curl_exec($ch);
        
        if (curl_error($ch)) {
            Log::error('PayTR cURL Error: ' . curl_error($ch));
            curl_close($ch);
            return ['status' => 'failed', 'reason' => 'Bağlantı hatası'];
        }
        
        curl_close($ch);
        
        $response = json_decode($result, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            Log::error('PayTR JSON Parse Error: ' . json_last_error_msg());
            return ['status' => 'failed', 'reason' => 'Yanıt formatı hatası'];
        }

        return $response;
    }

    /**
     * Test kartları bilgisi
     */
    public static function getTestCards(): array
    {
        return [
            'success' => [
                'number' => '4355 0841 0000 0001',
                'cvv' => '123',
                'expiry' => '12/26',
                'description' => 'Başarılı test kartı'
            ],
            'fail' => [
                'number' => '4355 0841 0000 0002',
                'cvv' => '123',
                'expiry' => '12/26',
                'description' => 'Başarısız test kartı'
            ]
        ];
    }

    /**
     * PayTR durumu kontrol et
     */
    public function getStatus(): array
    {
        return [
            'enabled' => $this->enabled,
            'test_mode' => $this->testMode,
            'configured' => $this->isEnabled(),
            'merchant_id' => $this->merchantId ? 'Tanımlı' : 'Tanımsız',
            'merchant_key' => $this->merchantKey ? 'Tanımlı' : 'Tanımsız',
            'merchant_salt' => $this->merchantSalt ? 'Tanımlı' : 'Tanımsız',
        ];
    }
}
