<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Story extends Model
{
    protected $fillable = [
        'title',
        'description',
        'image',
        'type',
        'product_id',
        'category_id',
        'custom_link',
        'background_color',
        'text_color',
        'sort_order',
        'is_active',
        'expires_at'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'expires_at' => 'datetime',
        'sort_order' => 'integer',
    ];

    /**
     * İlişkili ürün
     */
    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    /**
     * İlişkili kategori
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    /**
     * Scope: Sadece aktif story'ler
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true)
                    ->where(function($q) {
                        $q->whereNull('expires_at')
                          ->orWhere('expires_at', '>', now());
                    });
    }

    /**
     * Scope: Sıralı story'ler
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order')->orderBy('id');
    }

    /**
     * Story resim URL'si
     */
    public function getImageUrlAttribute()
    {
        if ($this->image) {
            return asset('storage/' . $this->image);
        }
        return null;
    }

    /**
     * Story link'ini al
     */
    public function getLinkAttribute()
    {
        switch ($this->type) {
            case 'product':
                return $this->product ? route('products.show', $this->product->id) : '#';
            case 'category':
                return $this->category ? route('categories.show', $this->category->id) : '#';
            case 'custom':
            case 'link':
                return $this->custom_link ?: '#';
            default:
                return '#';
        }
    }

    /**
     * Story'nin süresi dolmuş mu?
     */
    public function getIsExpiredAttribute()
    {
        return $this->expires_at && $this->expires_at->isPast();
    }

    /**
     * Story tipi Türkçe
     */
    public function getTypeTextAttribute()
    {
        return match($this->type) {
            'product' => 'Ürün',
            'category' => 'Kategori',
            'custom' => 'Özel',
            'link' => 'Link',
            default => 'Bilinmiyor'
        };
    }

    /**
     * Story durumu badge rengi
     */
    public function getStatusBadgeAttribute()
    {
        if (!$this->is_active) {
            return 'secondary';
        }

        if ($this->is_expired) {
            return 'warning';
        }

        return 'success';
    }

    /**
     * Story durumu metni
     */
    public function getStatusTextAttribute()
    {
        if (!$this->is_active) {
            return 'Pasif';
        }

        if ($this->is_expired) {
            return 'Süresi Dolmuş';
        }

        return 'Aktif';
    }
}
