<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Product extends Model
{
    protected $fillable = [
        'name',
        'slug',
        'description',
        'short_description',
        'features',
        'price',
        'sale_price',
        'discount_price',
        'stock_quantity',
        'min_stock_level',
        'sku',
        'image',
        'images',
        'featured_image',
        'is_featured',
        'is_active',
        'in_stock',
        'track_stock',
        'status',
        'attributes',
        'weight',
        'care_instructions',
        'delivery_days',
        'meta_title',
        'meta_description',
        'category_id'
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'sale_price' => 'decimal:2',
        'weight' => 'decimal:2',
        'is_featured' => 'boolean',
        'is_active' => 'boolean',
        'images' => 'array',
        'attributes' => 'array',
    ];

    /**
     * Ürünün kategorisi
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    /**
     * Sipariş öğeleri
     */
    public function orderItems(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    /**
     * Sepet öğeleri
     */
    public function cartItems(): HasMany
    {
        return $this->hasMany(CartItem::class);
    }

    /**
     * Aktual fiyat (indirimli fiyat varsa onu, yoksa normal fiyatı döndürür)
     */
    public function getActualPriceAttribute()
    {
        return $this->sale_price ?? $this->price;
    }

    /**
     * İndirim yüzdesi
     */
    public function getDiscountPercentageAttribute()
    {
        if (!$this->sale_price || $this->sale_price >= $this->price) {
            return 0;
        }

        return round((($this->price - $this->sale_price) / $this->price) * 100);
    }

    /**
     * Stokta var mı?
     */
    public function getInStockAttribute()
    {
        return $this->stock_quantity > 0;
    }

    /**
     * Ana resim URL'si
     */
    public function getMainImageAttribute()
    {
        if ($this->featured_image) {
            return asset('storage/' . $this->featured_image);
        }

        if ($this->images && count($this->images) > 0) {
            return asset('storage/' . $this->images[0]);
        }

        return asset('images/no-image.jpg'); // Varsayılan resim
    }

    /**
     * Scope: Sadece aktif ürünler
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true)->where('status', 'active');
    }

    /**
     * Scope: Öne çıkan ürünler
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    /**
     * Scope: Stokta olan ürünler
     */
    public function scopeInStock($query)
    {
        return $query->where('stock_quantity', '>', 0);
    }

    /**
     * Scope: İndirimli ürünler
     */
    public function scopeOnSale($query)
    {
        return $query->whereNotNull('sale_price')->where('sale_price', '<', 'price');
    }
}
