<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use App\Services\PayTRService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class CheckoutController extends Controller
{
    private $paytrService;

    public function __construct(PayTRService $paytrService)
    {
        $this->paytrService = $paytrService;
    }

    /**
     * Checkout sayfasını göster
     */
    public function index()
    {
        $cart = session('cart', []);

        if (empty($cart)) {
            return redirect()->route('cart.index')->with('error', 'Sepetiniz boş!');
        }

        // PayTR durumunu kontrol et
        $paytrEnabled = $this->paytrService->isEnabled();

        return view('checkout.index', compact('cart', 'paytrEnabled'));
    }

    /**
     * Siparişi işle ve kaydet
     */
    public function store(Request $request)
    {
        $request->validate([
            'delivery_name' => 'required|string|max:255',
            'delivery_phone' => 'required|string|max:20',
            'delivery_address' => 'required|string',
            'delivery_city' => 'required|string|max:100',
            'delivery_district' => 'required|string|max:100',
            'delivery_date' => 'required|date|after:today',
            'delivery_time' => 'nullable|string',
            'delivery_notes' => 'nullable|string',
            'payment_method' => 'required|in:cash,card,transfer',
            'notes' => 'nullable|string',
        ]);

        $cart = session('cart', []);

        if (empty($cart)) {
            return redirect()->route('cart.index')->with('error', 'Sepetiniz boş!');
        }

        DB::beginTransaction();

        try {
            // Sipariş toplamını hesapla
            $subtotal = 0;
            foreach ($cart as $id => $item) {
                $subtotal += $item['price'] * $item['quantity'];
            }

            $shippingCost = $subtotal >= 200 ? 0 : 15;
            $total = $subtotal + $shippingCost;

            // Sipariş oluştur
            $order = Order::create([
                'order_number' => Order::generateOrderNumber(),
                'user_id' => Auth::id(),
                'status' => 'pending',
                'subtotal' => $subtotal,
                'shipping_cost' => $shippingCost,
                'total_amount' => $total,
                'delivery_name' => $request->delivery_name,
                'delivery_phone' => $request->delivery_phone,
                'delivery_address' => $request->delivery_address,
                'delivery_city' => $request->delivery_city,
                'delivery_district' => $request->delivery_district,
                'delivery_date' => $request->delivery_date,
                'delivery_time' => $request->delivery_time,
                'delivery_notes' => $request->delivery_notes,
                'payment_method' => $request->payment_method,
                'notes' => $request->notes,
            ]);

            // Sipariş öğelerini oluştur ve stok güncelle
            foreach ($cart as $productId => $item) {
                $product = Product::findOrFail($productId);

                // Stok kontrolü
                if ($product->stock_quantity < $item['quantity']) {
                    throw new \Exception("'{$product->name}' ürünü için yeterli stok bulunmuyor!");
                }

                // Sipariş öğesi oluştur
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $productId,
                    'product_name' => $product->name,
                    'product_sku' => $product->sku,
                    'unit_price' => $item['price'],
                    'quantity' => $item['quantity'],
                    'total_price' => $item['price'] * $item['quantity'],
                ]);

                // Stok güncelle
                $product->decrement('stock_quantity', $item['quantity']);
            }

            DB::commit();

            // Sepeti temizle
            session()->forget('cart');

            // Kredi kartı ödemesi seçildiyse PayTR'ye yönlendir
            if ($request->payment_method === 'card' && $this->paytrService->isEnabled()) {
                return redirect()->route('payment.credit-card', ['order_id' => $order->id])
                               ->with('success', 'Siparişiniz oluşturuldu. Ödeme sayfasına yönlendiriliyorsunuz...');
            }

            return redirect()->route('checkout.success', $order->order_number)
                           ->with('success', 'Siparişiniz başarıyla alındı!');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->with('error', 'Sipariş işlenirken bir hata oluştu: ' . $e->getMessage());
        }
    }

    /**
     * Sipariş başarı sayfası
     */
    public function success($orderNumber)
    {
        $order = Order::where('order_number', $orderNumber)
                     ->where('user_id', Auth::id())
                     ->with('items.product')
                     ->firstOrFail();

        return view('checkout.success', compact('order'));
    }
}
