<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class CartController extends Controller
{
    /**
     * Sepet sayfası
     */
    public function index()
    {
        $cart = session('cart', []);

        // Mevcut sepetteki ürünler için eksik bilgileri tamamla
        foreach ($cart as $productId => $item) {
            $needsUpdate = false;
            $product = null;

            // Eksik bilgileri kontrol et
            if (!isset($item['category_icon']) || !isset($item['category_color']) || !isset($item['image'])) {
                $product = \App\Models\Product::with('category')->find($productId);
                $needsUpdate = true;
            }

            if ($needsUpdate && $product) {
                $cart[$productId]['category_icon'] = $product->category->icon ?? 'bi bi-flower1';
                $cart[$productId]['category_color'] = $product->category->color ?? '#E91E63';
                $cart[$productId]['image'] = $product->image;

                // Ürün adını ve fiyatını da güncelle (değişmiş olabilir)
                $cart[$productId]['name'] = $product->name;
                $cart[$productId]['price'] = $product->actual_price;
            }
        }

        // Güncellenmiş sepeti session'a kaydet
        session(['cart' => $cart]);

        return view('cart.index', compact('cart'));
    }

    /**
     * Sepete ürün ekleme
     */
    public function add(Request $request, $productId)
    {
        $product = \App\Models\Product::findOrFail($productId);
        $quantity = $request->get('quantity', 1);

        // Stok kontrolü
        if (!$product->in_stock || $product->stock_quantity < $quantity) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ürün stokta yok veya yeterli stok bulunmuyor!'
                ]);
            }
            return back()->with('error', 'Ürün stokta yok veya yeterli stok bulunmuyor!');
        }

        $cart = session('cart', []);

        if (isset($cart[$productId])) {
            $newQuantity = $cart[$productId]['quantity'] + $quantity;
            if ($newQuantity > $product->stock_quantity) {
                if ($request->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Stok miktarını aştınız!'
                    ]);
                }
                return back()->with('error', 'Stok miktarını aştınız!');
            }
            $cart[$productId]['quantity'] = $newQuantity;
        } else {
            $cart[$productId] = [
                'quantity' => $quantity,
                'name' => $product->name,
                'price' => $product->actual_price,
                'image' => $product->image,
                'category_icon' => $product->category->icon,
                'category_color' => $product->category->color
            ];
        }

        session(['cart' => $cart]);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Ürün sepete eklendi!',
                'cart_count' => count($cart)
            ]);
        }

        return back()->with('success', 'Ürün sepete eklendi!');
    }

    /**
     * Sepet ürün miktarını güncelleme
     */
    public function update(Request $request, $productId)
    {
        $product = \App\Models\Product::findOrFail($productId);
        $cart = session('cart', []);
        $quantity = $request->get('quantity', 1);

        if ($quantity <= 0) {
            unset($cart[$productId]);
            session(['cart' => $cart]);
            return back()->with('success', 'Ürün sepetten çıkarıldı!');
        }

        // Stok kontrolü
        if ($quantity > $product->stock_quantity) {
            return back()->with('error', 'Stok miktarını aştınız! Maksimum ' . $product->stock_quantity . ' adet ekleyebilirsiniz.');
        }

        if (isset($cart[$productId])) {
            $cart[$productId]['quantity'] = $quantity;
            session(['cart' => $cart]);
            return back()->with('success', 'Sepet güncellendi!');
        }

        return back()->with('error', 'Ürün sepette bulunamadı!');
    }

    /**
     * Sepetten ürün çıkarma
     */
    public function remove($product)
    {
        $cart = session('cart', []);
        unset($cart[$product]);
        session(['cart' => $cart]);
        return back()->with('success', 'Ürün sepetten çıkarıldı!');
    }

    /**
     * Sepet sayısını döndür (AJAX için)
     */
    public function count()
    {
        $cart = session('cart', []);
        return response()->json(['count' => count($cart)]);
    }

    /**
     * Sepeti temizle
     */
    public function clear()
    {
        session()->forget('cart');
        return redirect()->route('cart.index')->with('success', 'Sepet temizlendi!');
    }
}
