<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Order;
use App\Models\User;
use App\Models\Category;
use App\Models\Slider;
use App\Models\Setting;
use App\Models\Story;
use App\Models\PaymentSetting;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class AdminController extends Controller
{

    /**
     * Admin dashboard
     */
    public function dashboard()
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        // İstatistikler
        $stats = [
            'products' => Product::count(),
            'orders' => Order::count(),
            'customers' => User::count(),
            'monthly_revenue' => Order::whereMonth('created_at', now()->month)
                                    ->whereYear('created_at', now()->year)
                                    ->sum('total_amount')
        ];

        // Son siparişler
        $recentOrders = Order::with('user')
                            ->latest()
                            ->take(10)
                            ->get();

        // Düşük stoklu ürünler
        $lowStockProducts = Product::where('stock_quantity', '<=', 5)
                                  ->where('stock_quantity', '>', 0)
                                  ->get();

        return view('admin.dashboard', compact('stats', 'recentOrders', 'lowStockProducts'));
    }

    /**
     * Ürün listeleme
     */
    public function products(Request $request)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $query = Product::with('category');

        // Arama
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('sku', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Kategori filtresi
        if ($request->filled('category')) {
            $query->where('category_id', $request->category);
        }

        // Durum filtresi
        if ($request->filled('status')) {
            switch ($request->status) {
                case 'active':
                    $query->where('is_active', true);
                    break;
                case 'inactive':
                    $query->where('is_active', false);
                    break;
                case 'low_stock':
                    $query->where('stock_quantity', '<=', 5);
                    break;
            }
        }

        // Sıralama
        $sort = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc');

        switch ($sort) {
            case 'name':
                $query->orderBy('name', $direction);
                break;
            case 'price':
                $query->orderBy('price', $direction);
                break;
            case 'stock':
                $query->orderBy('stock_quantity', $direction);
                break;
            default:
                $query->orderBy('created_at', $direction);
                break;
        }

        $products = $query->paginate(20);
        $categories = Category::orderBy('name')->get();

        return view('admin.products.index', compact('products', 'categories'));
    }

    /**
     * Ürün durumu değiştir
     */
    public function toggleProductStatus(Request $request, $productId)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            return response()->json(['success' => false, 'message' => 'Yetkisiz erişim']);
        }

        $product = Product::findOrFail($productId);
        $product->update(['is_active' => $request->is_active]);

        return response()->json(['success' => true]);
    }

    /**
     * Ürün ekleme formu
     */
    public function createProduct()
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $categories = Category::orderBy('name')->get();
        return view('admin.products.create', compact('categories'));
    }

    /**
     * Ürün kaydetme
     */
    public function storeProduct(Request $request)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'sku' => 'required|string|max:100|unique:products,sku',
            'description' => 'nullable|string',
            'features' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'price' => 'required|numeric|min:0',
            'discount_price' => 'nullable|numeric|min:0|lt:price',
            'stock_quantity' => 'required|integer|min:0',
            'min_stock_level' => 'nullable|integer|min:0',
            'category_id' => 'required|exists:categories,id',
            'slug' => 'nullable|string|max:255|unique:products,slug',
            'meta_title' => 'nullable|string|max:60',
            'meta_description' => 'nullable|string|max:160',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
            'in_stock' => 'boolean',
            'track_stock' => 'boolean',
        ]);

        // Slug oluştur
        $slug = $request->slug ?: \Str::slug($request->name);

        // Slug benzersizliği kontrolü
        $originalSlug = $slug;
        $counter = 1;
        while (Product::where('slug', $slug)->exists()) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        // Resim yükleme
        $imagePath = null;
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_' . $slug . '.' . $image->getClientOriginalExtension();
            $imagePath = $image->storeAs('products', $imageName, 'public');
        }

        // Features'ı array'e çevir
        $features = $request->features ? array_filter(explode("\n", $request->features)) : null;

        $product = Product::create([
            'name' => $request->name,
            'slug' => $slug,
            'sku' => $request->sku,
            'description' => $request->description,
            'features' => $features ? json_encode($features) : null,
            'image' => $imagePath,
            'price' => $request->price,
            'discount_price' => $request->discount_price,
            'stock_quantity' => $request->stock_quantity,
            'min_stock_level' => $request->min_stock_level ?? 5,
            'category_id' => $request->category_id,
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description,
            'is_active' => $request->boolean('is_active', true),
            'is_featured' => $request->boolean('is_featured', false),
            'in_stock' => $request->boolean('in_stock', true),
            'track_stock' => $request->boolean('track_stock', true),
        ]);

        return redirect()->route('admin.products.index')
                        ->with('success', 'Ürün başarıyla eklendi!');
    }

    /**
     * Ürün düzenleme formu
     */
    public function editProduct($productId)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $product = Product::with('category')->findOrFail($productId);
        $categories = Category::orderBy('name')->get();

        return view('admin.products.edit', compact('product', 'categories'));
    }

    /**
     * Ürün güncelleme
     */
    public function updateProduct(Request $request, $productId)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $product = Product::findOrFail($productId);

        $request->validate([
            'name' => 'required|string|max:255',
            'sku' => 'required|string|max:100|unique:products,sku,' . $product->id,
            'description' => 'nullable|string',
            'features' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'price' => 'required|numeric|min:0',
            'discount_price' => 'nullable|numeric|min:0|lt:price',
            'stock_quantity' => 'required|integer|min:0',
            'min_stock_level' => 'nullable|integer|min:0',
            'category_id' => 'required|exists:categories,id',
            'slug' => 'nullable|string|max:255|unique:products,slug,' . $product->id,
            'meta_title' => 'nullable|string|max:60',
            'meta_description' => 'nullable|string|max:160',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
            'in_stock' => 'boolean',
            'track_stock' => 'boolean',
            'remove_current_image' => 'boolean',
        ]);

        // Slug oluştur
        $slug = $request->slug ?: \Str::slug($request->name);

        // Slug benzersizliği kontrolü (mevcut ürün hariç)
        $originalSlug = $slug;
        $counter = 1;
        while (Product::where('slug', $slug)->where('id', '!=', $product->id)->exists()) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        // Resim işleme
        $imagePath = $product->image; // Mevcut resmi koru

        // Mevcut resmi sil seçeneği
        if ($request->boolean('remove_current_image')) {
            if ($product->image && \Storage::disk('public')->exists($product->image)) {
                \Storage::disk('public')->delete($product->image);
            }
            $imagePath = null;
        }

        // Yeni resim yükleme
        if ($request->hasFile('image')) {
            // Eski resmi sil
            if ($product->image && \Storage::disk('public')->exists($product->image)) {
                \Storage::disk('public')->delete($product->image);
            }

            $image = $request->file('image');
            $imageName = time() . '_' . $slug . '.' . $image->getClientOriginalExtension();
            $imagePath = $image->storeAs('products', $imageName, 'public');
        }

        // Features'ı array'e çevir
        $features = $request->features ? array_filter(explode("\n", $request->features)) : null;

        $product->update([
            'name' => $request->name,
            'slug' => $slug,
            'sku' => $request->sku,
            'description' => $request->description,
            'features' => $features ? json_encode($features) : null,
            'image' => $imagePath,
            'price' => $request->price,
            'discount_price' => $request->discount_price,
            'stock_quantity' => $request->stock_quantity,
            'min_stock_level' => $request->min_stock_level ?? 5,
            'category_id' => $request->category_id,
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description,
            'is_active' => $request->boolean('is_active', false),
            'is_featured' => $request->boolean('is_featured', false),
            'in_stock' => $request->boolean('in_stock', false),
            'track_stock' => $request->boolean('track_stock', true),
        ]);

        return redirect()->route('admin.products.index')
                        ->with('success', 'Ürün başarıyla güncellendi!');
    }

    /**
     * Ürün silme
     */
    public function destroyProduct($productId)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $product = Product::findOrFail($productId);

        // Sipariş kontrolü - eğer bu ürünle ilgili siparişler varsa silinmesine izin verme
        $orderItemsCount = \App\Models\OrderItem::where('product_id', $product->id)->count();

        if ($orderItemsCount > 0) {
            return redirect()->route('admin.products.index')
                           ->with('error', 'Bu ürün siparişlerde kullanıldığı için silinemez! Bunun yerine ürünü pasif yapabilirsiniz.');
        }

        $productName = $product->name;
        $product->delete();

        return redirect()->route('admin.products.index')
                        ->with('success', "'{$productName}' ürünü başarıyla silindi!");
    }

    /**
     * Toplu durum güncelleme
     */
    public function bulkUpdateProductStatus(Request $request)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            return response()->json(['success' => false, 'message' => 'Yetkisiz erişim']);
        }

        $request->validate([
            'product_ids' => 'required|array',
            'product_ids.*' => 'exists:products,id',
            'is_active' => 'required|boolean'
        ]);

        $updatedCount = Product::whereIn('id', $request->product_ids)
                              ->update(['is_active' => $request->is_active]);

        return response()->json([
            'success' => true,
            'message' => "{$updatedCount} ürün başarıyla güncellendi!"
        ]);
    }

    /**
     * Toplu silme
     */
    public function bulkDeleteProducts(Request $request)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            return response()->json(['success' => false, 'message' => 'Yetkisiz erişim']);
        }

        $request->validate([
            'product_ids' => 'required|array',
            'product_ids.*' => 'exists:products,id'
        ]);

        // Sipariş kontrolü
        $productsWithOrders = \App\Models\OrderItem::whereIn('product_id', $request->product_ids)
                                                  ->pluck('product_id')
                                                  ->unique()
                                                  ->toArray();

        if (!empty($productsWithOrders)) {
            $productNames = Product::whereIn('id', $productsWithOrders)->pluck('name')->toArray();
            return response()->json([
                'success' => false,
                'message' => 'Şu ürünler siparişlerde kullanıldığı için silinemez: ' . implode(', ', $productNames)
            ]);
        }

        $deletedCount = Product::whereIn('id', $request->product_ids)->delete();

        return response()->json([
            'success' => true,
            'message' => "{$deletedCount} ürün başarıyla silindi!"
        ]);
    }

    /**
     * Sipariş listeleme
     */
    public function orders(Request $request)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $query = Order::with(['user', 'items']);

        // Arama
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('order_number', 'like', "%{$search}%")
                  ->orWhereHas('user', function($userQuery) use ($search) {
                      $userQuery->where('name', 'like', "%{$search}%")
                               ->orWhere('email', 'like', "%{$search}%");
                  })
                  ->orWhere('delivery_name', 'like', "%{$search}%")
                  ->orWhere('delivery_phone', 'like', "%{$search}%");
            });
        }

        // Durum filtresi
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Ödeme yöntemi filtresi
        if ($request->filled('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        // Tarih filtresi
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Sıralama
        $query->orderBy('created_at', 'desc');

        $orders = $query->paginate(20);

        // İstatistikler
        $stats = [
            'pending' => Order::where('status', 'pending')->count(),
            'confirmed' => Order::where('status', 'confirmed')->count(),
            'shipped' => Order::where('status', 'shipped')->count(),
            'delivered' => Order::where('status', 'delivered')->count(),
        ];

        return view('admin.orders.index', compact('orders', 'stats'));
    }

    /**
     * Sipariş durumu güncelleme
     */
    public function updateOrderStatus(Request $request, $orderId)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            return response()->json(['success' => false, 'message' => 'Yetkisiz erişim']);
        }

        $request->validate([
            'status' => 'required|in:pending,confirmed,shipped,delivered,cancelled'
        ]);

        $order = Order::findOrFail($orderId);

        // Eğer sipariş iptal ediliyorsa stokları geri ekle
        if ($request->status === 'cancelled' && $order->status !== 'cancelled') {
            foreach ($order->items as $item) {
                $item->product->increment('stock_quantity', $item->quantity);
            }
        }

        // Eğer iptal edilmiş sipariş tekrar aktif hale getiriliyorsa stokları düş
        if ($order->status === 'cancelled' && $request->status !== 'cancelled') {
            foreach ($order->items as $item) {
                if ($item->product->stock_quantity < $item->quantity) {
                    return response()->json([
                        'success' => false,
                        'message' => "'{$item->product->name}' ürünü için yeterli stok yok!"
                    ]);
                }
                $item->product->decrement('stock_quantity', $item->quantity);
            }
        }

        $order->update(['status' => $request->status]);

        return response()->json(['success' => true]);
    }

    /**
     * Sipariş detayı
     */
    public function showOrder($orderId)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $order = Order::with(['user', 'items.product.category'])->findOrFail($orderId);

        return view('admin.orders.show', compact('order'));
    }

    /**
     * Kargo takip numarası güncelleme
     */
    public function updateOrderTracking(Request $request, $orderId)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            return response()->json(['success' => false, 'message' => 'Yetkisiz erişim']);
        }

        $request->validate([
            'tracking_number' => 'nullable|string|max:100'
        ]);

        $order = Order::findOrFail($orderId);
        $order->update(['tracking_number' => $request->tracking_number]);

        return response()->json(['success' => true]);
    }

    /**
     * Kategori listeleme
     */
    public function categories()
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $categories = Category::withCount('products')->orderBy('sort_order')->get();

        return view('admin.categories.index', compact('categories'));
    }

    /**
     * Kategori ekleme formu
     */
    public function createCategory()
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        return view('admin.categories.create');
    }

    /**
     * Kategori kaydetme
     */
    public function storeCategory(Request $request)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'color' => 'required|string|max:7',
            'icon' => 'required|string|max:100',
            'sort_order' => 'nullable|integer|min:0',
            'slug' => 'nullable|string|max:255|unique:categories,slug',
            'meta_title' => 'nullable|string|max:60',
            'meta_description' => 'nullable|string|max:160',
            'is_active' => 'boolean',
            'show_on_homepage' => 'boolean',
        ]);

        // Slug oluştur
        $slug = $request->slug ?: \Str::slug($request->name);

        // Slug benzersizliği kontrolü
        $originalSlug = $slug;
        $counter = 1;
        while (Category::where('slug', $slug)->exists()) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        // Resim yükleme
        $imagePath = null;
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_' . $slug . '.' . $image->getClientOriginalExtension();
            $imagePath = $image->storeAs('categories', $imageName, 'public');
        }

        Category::create([
            'name' => $request->name,
            'slug' => $slug,
            'description' => $request->description,
            'image' => $imagePath,
            'color' => $request->color,
            'icon' => $request->icon,
            'sort_order' => $request->sort_order ?? 0,
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description,
            'is_active' => $request->boolean('is_active', true),
            'show_on_homepage' => $request->boolean('show_on_homepage', true),
        ]);

        return redirect()->route('admin.categories.index')
                        ->with('success', 'Kategori başarıyla eklendi!');
    }

    /**
     * Kategori durumu değiştir
     */
    public function toggleCategoryStatus(Request $request, $categoryId)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            return response()->json(['success' => false, 'message' => 'Yetkisiz erişim']);
        }

        $category = Category::findOrFail($categoryId);
        $category->update(['is_active' => $request->is_active]);

        return response()->json(['success' => true]);
    }

    /**
     * Kategori düzenleme formu
     */
    public function editCategory($categoryId)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $category = Category::findOrFail($categoryId);
        return view('admin.categories.edit', compact('category'));
    }

    /**
     * Kategori güncelleme
     */
    public function updateCategory(Request $request, $categoryId)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $category = Category::findOrFail($categoryId);

        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'color' => 'required|string|max:7',
            'icon' => 'required|string|max:100',
            'sort_order' => 'nullable|integer|min:0',
            'slug' => 'nullable|string|max:255|unique:categories,slug,' . $category->id,
            'meta_title' => 'nullable|string|max:60',
            'meta_description' => 'nullable|string|max:160',
            'is_active' => 'boolean',
            'show_on_homepage' => 'boolean',
            'remove_current_image' => 'boolean',
        ]);

        // Slug oluştur
        $slug = $request->slug ?: \Str::slug($request->name);

        // Slug benzersizliği kontrolü (mevcut kategori hariç)
        $originalSlug = $slug;
        $counter = 1;
        while (Category::where('slug', $slug)->where('id', '!=', $category->id)->exists()) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        // Resim işleme
        $imagePath = $category->image; // Mevcut resmi koru

        // Mevcut resmi sil seçeneği
        if ($request->boolean('remove_current_image')) {
            if ($category->image && \Storage::disk('public')->exists($category->image)) {
                \Storage::disk('public')->delete($category->image);
            }
            $imagePath = null;
        }

        // Yeni resim yükleme
        if ($request->hasFile('image')) {
            // Eski resmi sil
            if ($category->image && \Storage::disk('public')->exists($category->image)) {
                \Storage::disk('public')->delete($category->image);
            }

            $image = $request->file('image');
            $imageName = time() . '_' . $slug . '.' . $image->getClientOriginalExtension();
            $imagePath = $image->storeAs('categories', $imageName, 'public');
        }

        $category->update([
            'name' => $request->name,
            'slug' => $slug,
            'description' => $request->description,
            'image' => $imagePath,
            'color' => $request->color,
            'icon' => $request->icon,
            'sort_order' => $request->sort_order ?? 0,
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description,
            'is_active' => $request->boolean('is_active', false),
            'show_on_homepage' => $request->boolean('show_on_homepage', false),
        ]);

        return redirect()->route('admin.categories.index')
                        ->with('success', 'Kategori başarıyla güncellendi!');
    }

    /**
     * Kategori silme
     */
    public function destroyCategory($categoryId)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $category = Category::withCount('products')->findOrFail($categoryId);

        // Ürün kontrolü - eğer bu kategoride ürünler varsa uyarı ver
        if ($category->products_count > 0) {
            // Ürünleri kategorisiz yap
            Product::where('category_id', $category->id)->update(['category_id' => null]);
        }

        $categoryName = $category->name;

        // Kategori resmini sil
        if ($category->image && \Storage::disk('public')->exists($category->image)) {
            \Storage::disk('public')->delete($category->image);
        }

        $category->delete();

        return redirect()->route('admin.categories.index')
                        ->with('success', "'{$categoryName}' kategorisi başarıyla silindi!" .
                               ($category->products_count > 0 ? " {$category->products_count} ürün kategorisiz kaldı." : ""));
    }

    /**
     * Raporlar sayfası
     */
    public function reports()
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        // Genel istatistikler
        $totalProducts = Product::count();
        $activeProducts = Product::where('is_active', true)->count();
        $totalCategories = Category::count();
        $totalOrders = Order::count();
        $totalUsers = User::count();
        $totalRevenue = Order::where('status', 'delivered')->sum('total_amount');

        // Bu ay istatistikleri
        $thisMonth = now()->startOfMonth();
        $monthlyOrders = Order::where('created_at', '>=', $thisMonth)->count();
        $monthlyRevenue = Order::where('created_at', '>=', $thisMonth)
                              ->where('status', 'delivered')
                              ->sum('total_amount');
        $monthlyUsers = User::where('created_at', '>=', $thisMonth)->count();

        // Son 7 gün satış grafiği
        $salesChart = [];
        for ($i = 6; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $dailySales = Order::whereDate('created_at', $date)
                             ->where('status', 'delivered')
                             ->sum('total_amount');
            $salesChart[] = [
                'date' => $date->format('d.m'),
                'sales' => $dailySales
            ];
        }

        // En çok satan ürünler
        $topProducts = Product::withCount(['orderItems' => function($query) {
            $query->whereHas('order', function($q) {
                $q->where('status', 'delivered');
            });
        }])
        ->orderBy('order_items_count', 'desc')
        ->limit(10)
        ->get();

        // Kategori bazında satışlar
        $categorySales = Category::withCount(['products' => function($query) {
            $query->withCount(['orderItems' => function($q) {
                $q->whereHas('order', function($order) {
                    $order->where('status', 'delivered');
                });
            }]);
        }])
        ->with(['products' => function($query) {
            $query->withCount(['orderItems' => function($q) {
                $q->whereHas('order', function($order) {
                    $order->where('status', 'delivered');
                });
            }]);
        }])
        ->get()
        ->map(function($category) {
            $totalSales = $category->products->sum('order_items_count');
            return [
                'name' => $category->name,
                'color' => $category->color,
                'sales' => $totalSales,
                'products_count' => $category->products_count
            ];
        })
        ->sortByDesc('sales')
        ->take(8);

        // Son siparişler
        $recentOrders = Order::with(['user', 'items.product'])
                           ->orderBy('created_at', 'desc')
                           ->limit(10)
                           ->get();

        // Sipariş durumları
        $orderStatuses = Order::selectRaw('status, COUNT(*) as count')
                            ->groupBy('status')
                            ->get()
                            ->pluck('count', 'status');

        return view('admin.reports.index', compact(
            'totalProducts', 'activeProducts', 'totalCategories', 'totalOrders',
            'totalUsers', 'totalRevenue', 'monthlyOrders', 'monthlyRevenue',
            'monthlyUsers', 'salesChart', 'topProducts', 'categorySales',
            'recentOrders', 'orderStatuses'
        ));
    }

    /**
     * Slider yönetimi
     */
    public function sliders()
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $sliders = Slider::ordered()->get();

        return view('admin.sliders.index', compact('sliders'));
    }

    /**
     * Slider ekleme formu
     */
    public function createSlider()
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        return view('admin.sliders.create');
    }

    /**
     * Slider kaydetme
     */
    public function storeSlider(Request $request)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'required|image|mimes:jpeg,png,jpg,webp|max:2048',
            'button_text' => 'nullable|string|max:100',
            'button_link' => 'nullable|url',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        $data = $request->only(['title', 'description', 'button_text', 'button_link', 'sort_order']);

        // Resim yükleme
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
            $imagePath = $image->storeAs('sliders', $imageName, 'public');
            $data['image'] = $imagePath;
        }

        $data['is_active'] = $request->boolean('is_active', true);
        $data['sort_order'] = $data['sort_order'] ?? 0;

        Slider::create($data);

        return redirect()->route('admin.sliders.index')
                        ->with('success', 'Slider başarıyla eklendi!');
    }

    /**
     * Slider düzenleme formu
     */
    public function editSlider(Slider $slider)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        return view('admin.sliders.edit', compact('slider'));
    }

    /**
     * Slider güncelleme
     */
    public function updateSlider(Request $request, Slider $slider)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'button_text' => 'nullable|string|max:100',
            'button_link' => 'nullable|url',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        $data = $request->only(['title', 'description', 'button_text', 'button_link', 'sort_order']);

        // Resim güncelleme
        if ($request->hasFile('image')) {
            // Eski resmi sil
            if ($slider->image && Storage::disk('public')->exists($slider->image)) {
                Storage::disk('public')->delete($slider->image);
            }

            $image = $request->file('image');
            $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
            $imagePath = $image->storeAs('sliders', $imageName, 'public');
            $data['image'] = $imagePath;
        }

        $data['is_active'] = $request->boolean('is_active', false);
        $data['sort_order'] = $data['sort_order'] ?? 0;

        $slider->update($data);

        return redirect()->route('admin.sliders.index')
                        ->with('success', 'Slider başarıyla güncellendi!');
    }

    /**
     * Slider silme
     */
    public function destroySlider(Slider $slider)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $sliderTitle = $slider->title;

        // Resmi sil
        if ($slider->image && Storage::disk('public')->exists($slider->image)) {
            Storage::disk('public')->delete($slider->image);
        }

        $slider->delete();

        return redirect()->route('admin.sliders.index')
                        ->with('success', "'{$sliderTitle}' slider'ı başarıyla silindi!");
    }

    /**
     * Slider durumu değiştir
     */
    public function toggleSliderStatus(Request $request, Slider $slider)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            return response()->json(['success' => false, 'message' => 'Yetkiniz yok'], 403);
        }

        $slider->update([
            'is_active' => $request->boolean('is_active')
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Slider durumu güncellendi',
            'is_active' => $slider->is_active
        ]);
    }

    /**
     * Site ayarları
     */
    public function settings()
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $settingGroups = [
            'general' => 'Genel Ayarlar',
            'seo' => 'SEO Ayarları',
            'business' => 'İşletme Bilgileri',
            'social' => 'Sosyal Medya',
            'ecommerce' => 'E-ticaret Ayarları',
            'footer' => 'Footer Ayarları',
            'pages' => 'Sayfa Yönetimi'
        ];

        $settings = [];
        foreach ($settingGroups as $group => $title) {
            $settings[$group] = [
                'title' => $title,
                'settings' => Setting::byGroup($group)->ordered()->get()
            ];
        }

        return view('admin.settings.index', compact('settings', 'settingGroups'));
    }

    /**
     * Site ayarlarını güncelle
     */
    public function updateSettings(Request $request)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $settings = $request->input('settings', []);

        foreach ($settings as $key => $value) {
            $setting = Setting::where('key', $key)->first();

            if ($setting) {
                // Resim yükleme kontrolü
                if ($setting->type === 'image' && $request->hasFile("settings.{$key}")) {
                    $file = $request->file("settings.{$key}");

                    // Eski resmi sil
                    if ($setting->value && Storage::disk('public')->exists($setting->value)) {
                        Storage::disk('public')->delete($setting->value);
                    }

                    // Yeni resmi kaydet
                    $fileName = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
                    $filePath = $file->storeAs('settings', $fileName, 'public');
                    $value = $filePath;
                }

                // Boolean değer kontrolü
                if ($setting->type === 'boolean') {
                    $value = $request->boolean("settings.{$key}") ? '1' : '0';
                }

                $setting->update(['value' => $value]);
            }
        }

        // Cache'i temizle
        Setting::clearCache();

        return redirect()->route('admin.settings.index')
                        ->with('success', 'Site ayarları başarıyla güncellendi!');
    }

    /**
     * Story yönetimi
     */
    public function stories()
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $stories = Story::with(['product', 'category'])->ordered()->get();

        return view('admin.stories.index', compact('stories'));
    }

    /**
     * Story ekleme formu
     */
    public function createStory()
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $products = Product::select('id', 'name')->get();
        $categories = Category::select('id', 'name')->get();

        return view('admin.stories.create', compact('products', 'categories'));
    }

    /**
     * Story kaydetme
     */
    public function storeStory(Request $request)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'required|image|mimes:jpeg,png,jpg,webp|max:2048',
            'type' => 'required|in:product,category,custom,link',
            'product_id' => 'nullable|exists:products,id',
            'category_id' => 'nullable|exists:categories,id',
            'custom_link' => 'nullable|url',
            'background_color' => 'required|string',
            'text_color' => 'required|string',
            'sort_order' => 'nullable|integer|min:0',
            'expires_at' => 'nullable|date|after:now',
        ]);

        $data = $request->only([
            'title', 'description', 'type', 'product_id', 'category_id',
            'custom_link', 'background_color', 'text_color', 'sort_order', 'expires_at'
        ]);

        // Resim yükleme
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
            $imagePath = $image->storeAs('stories', $imageName, 'public');
            $data['image'] = $imagePath;
        }

        // Tip kontrolü
        if ($data['type'] === 'product' && !$data['product_id']) {
            return back()->withErrors(['product_id' => 'Ürün story\'si için ürün seçmelisiniz.']);
        }

        if ($data['type'] === 'category' && !$data['category_id']) {
            return back()->withErrors(['category_id' => 'Kategori story\'si için kategori seçmelisiniz.']);
        }

        if (in_array($data['type'], ['custom', 'link']) && !$data['custom_link']) {
            return back()->withErrors(['custom_link' => 'Özel story için link girmelisiniz.']);
        }

        $data['is_active'] = $request->boolean('is_active', true);
        $data['sort_order'] = $data['sort_order'] ?? 0;

        Story::create($data);

        return redirect()->route('admin.stories.index')
                        ->with('success', 'Story başarıyla eklendi!');
    }

    /**
     * Story düzenleme formu
     */
    public function editStory(Story $story)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $products = Product::select('id', 'name')->get();
        $categories = Category::select('id', 'name')->get();

        return view('admin.stories.edit', compact('story', 'products', 'categories'));
    }

    /**
     * Story güncelleme
     */
    public function updateStory(Request $request, Story $story)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'type' => 'required|in:product,category,custom,link',
            'product_id' => 'nullable|exists:products,id',
            'category_id' => 'nullable|exists:categories,id',
            'custom_link' => 'nullable|url',
            'background_color' => 'required|string',
            'text_color' => 'required|string',
            'sort_order' => 'nullable|integer|min:0',
            'expires_at' => 'nullable|date|after:now',
        ]);

        $data = $request->only([
            'title', 'description', 'type', 'product_id', 'category_id',
            'custom_link', 'background_color', 'text_color', 'sort_order', 'expires_at'
        ]);

        // Resim güncelleme
        if ($request->hasFile('image')) {
            // Eski resmi sil
            if ($story->image && Storage::disk('public')->exists($story->image)) {
                Storage::disk('public')->delete($story->image);
            }

            $image = $request->file('image');
            $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
            $imagePath = $image->storeAs('stories', $imageName, 'public');
            $data['image'] = $imagePath;
        }

        // Tip kontrolü
        if ($data['type'] === 'product' && !$data['product_id']) {
            return back()->withErrors(['product_id' => 'Ürün story\'si için ürün seçmelisiniz.']);
        }

        if ($data['type'] === 'category' && !$data['category_id']) {
            return back()->withErrors(['category_id' => 'Kategori story\'si için kategori seçmelisiniz.']);
        }

        if (in_array($data['type'], ['custom', 'link']) && !$data['custom_link']) {
            return back()->withErrors(['custom_link' => 'Özel story için link girmelisiniz.']);
        }

        $data['is_active'] = $request->boolean('is_active', false);
        $data['sort_order'] = $data['sort_order'] ?? 0;

        $story->update($data);

        return redirect()->route('admin.stories.index')
                        ->with('success', 'Story başarıyla güncellendi!');
    }

    /**
     * Story silme
     */
    public function destroyStory(Story $story)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $storyTitle = $story->title;

        // Resmi sil
        if ($story->image && Storage::disk('public')->exists($story->image)) {
            Storage::disk('public')->delete($story->image);
        }

        $story->delete();

        return redirect()->route('admin.stories.index')
                        ->with('success', "'{$storyTitle}' story'si başarıyla silindi!");
    }

    /**
     * Story durumu değiştir
     */
    public function toggleStoryStatus(Request $request, Story $story)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            return response()->json(['success' => false, 'message' => 'Yetkiniz yok'], 403);
        }

        $story->update([
            'is_active' => $request->boolean('is_active')
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Story durumu güncellendi',
            'is_active' => $story->is_active
        ]);
    }

    /**
     * PayTR Ödeme Ayarları
     */
    public function paymentSettings()
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $settings = PaymentSetting::getByGroup('paytr');

        return view('admin.payment-settings', compact('settings'));
    }

    /**
     * PayTR Ödeme Ayarlarını Güncelle
     */
    public function updatePaymentSettings(Request $request)
    {
        // Admin kontrolü
        if (!Auth::check() || !Auth::user()->isAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $request->validate([
            'paytr_enabled' => 'boolean',
            'paytr_test_mode' => 'boolean',
            'paytr_merchant_id' => 'required_if:paytr_enabled,1|string|max:255',
            'paytr_merchant_key' => 'required_if:paytr_enabled,1|string|max:255',
            'paytr_merchant_salt' => 'required_if:paytr_enabled,1|string|max:255',
        ]);

        // PayTR ayarlarını güncelle
        PaymentSetting::set('paytr_enabled', $request->boolean('paytr_enabled'));
        PaymentSetting::set('paytr_test_mode', $request->boolean('paytr_test_mode'));
        PaymentSetting::set('paytr_merchant_id', $request->input('paytr_merchant_id', ''));
        PaymentSetting::set('paytr_merchant_key', $request->input('paytr_merchant_key', ''));
        PaymentSetting::set('paytr_merchant_salt', $request->input('paytr_merchant_salt', ''));

        return redirect()->route('admin.payment-settings')
                        ->with('success', 'PayTR ödeme ayarları başarıyla güncellendi!');
    }
}
