<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Order extends Model
{
    protected $fillable = [
        'order_number',
        'user_id',
        'status',
        'tracking_number',
        'subtotal',
        'tax_amount',
        'shipping_cost',
        'discount_amount',
        'total_amount',
        'delivery_name',
        'delivery_phone',
        'delivery_address',
        'delivery_city',
        'delivery_district',
        'delivery_postal_code',
        'delivery_date',
        'delivery_time',
        'delivery_notes',
        'payment_method',
        'payment_status',
        'paid_at',
        'notes',
        'payment_token',
        'paytr_merchant_oid',
        'payment_response',
        'payment_attempted_at'
    ];

    protected $casts = [
        'subtotal' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'shipping_cost' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'delivery_date' => 'date',
        'paid_at' => 'datetime',
        'payment_response' => 'array',
        'payment_attempted_at' => 'datetime',
    ];

    /**
     * Siparişi veren kullanıcı
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Sipariş öğeleri
     */
    public function items(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    /**
     * Sipariş numarası oluştur
     */
    public static function generateOrderNumber()
    {
        $prefix = 'CB'; // Çiçek Bahçesi
        $date = now()->format('Ymd');
        $lastOrder = self::whereDate('created_at', today())->count();
        $sequence = str_pad($lastOrder + 1, 4, '0', STR_PAD_LEFT);

        return $prefix . $date . $sequence;
    }

    /**
     * Sipariş durumu badge rengi
     */
    public function getStatusBadgeAttribute()
    {
        return match($this->status) {
            'pending' => 'warning',
            'confirmed' => 'info',
            'shipped' => 'primary',
            'delivered' => 'success',
            'cancelled' => 'danger',
            default => 'secondary'
        };
    }

    /**
     * Sipariş durumu Türkçe
     */
    public function getStatusTextAttribute()
    {
        return match($this->status) {
            'pending' => 'Beklemede',
            'confirmed' => 'Onaylandı',
            'shipped' => 'Kargoya Verildi',
            'delivered' => 'Teslim Edildi',
            'cancelled' => 'İptal Edildi',
            default => 'Bilinmiyor'
        };
    }

    /**
     * Scope: Belirli durumda olan siparişler
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }
}
