<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Category;

class ProductController extends Controller
{
    /**
     * Ürün listeleme sayfası
     */
    public function index(Request $request)
    {
        $query = Product::active()
            ->inStock()
            ->with('category');

        // Kategori filtresi
        if ($request->filled('category')) {
            $category = Category::where('slug', $request->category)->first();
            if ($category) {
                $query->where('category_id', $category->id);
            }
        }

        // Fiyat filtresi
        if ($request->filled('min_price')) {
            $query->where('price', '>=', $request->min_price);
        }

        if ($request->filled('max_price')) {
            $query->where('price', '<=', $request->max_price);
        }

        // Sıralama
        switch ($request->get('sort', 'name')) {
            case 'price_asc':
                $query->orderBy('price', 'asc');
                break;
            case 'price_desc':
                $query->orderBy('price', 'desc');
                break;
            case 'newest':
                $query->latest();
                break;
            default:
                $query->orderBy('name');
        }

        $products = $query->paginate(12)->withQueryString();

        // Kategorileri filtre için çek
        $categories = Category::active()->main()->ordered()->get();

        return view('products.index', compact('products', 'categories'));
    }

    /**
     * Ürün detay sayfası
     */
    public function show($productSlug)
    {
        $product = Product::where('slug', $productSlug)
            ->active()
            ->with('category')
            ->firstOrFail();

        // İlgili ürünler (aynı kategoriden)
        $relatedProducts = Product::active()
            ->inStock()
            ->where('category_id', $product->category_id)
            ->where('id', '!=', $product->id)
            ->take(4)
            ->get();

        return view('products.show', compact('product', 'relatedProducts'));
    }

    /**
     * Ürün arama
     */
    public function search(Request $request)
    {
        $query = $request->get('q');
        $products = collect();

        if ($query) {
            $products = Product::active()
                ->inStock()
                ->with('category')
                ->where(function($q) use ($query) {
                    $q->where('name', 'LIKE', "%{$query}%")
                      ->orWhere('description', 'LIKE', "%{$query}%")
                      ->orWhere('short_description', 'LIKE', "%{$query}%");
                })
                ->paginate(12)
                ->withQueryString();
        }

        return view('products.search', compact('query', 'products'));
    }
}
